<?php

/**
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2021 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

namespace NRFramework\Widgets;

defined('_JEXEC') or die;

/**
 * Countdown
 */
class Countdown extends Widget
{
	/**
	 * Widget default options
	 *
	 * @var array
	 */
	protected $widget_options = [
		/**
		 * The Countdown type:
		 * 
		 * - static: Counts down to a specific date and time. Universal deadline for all visitors.
		 * - dynamic: Set-and-forget solution. The countdown starts when your visitor sees the offer.
		 */
		'countdown_type' => 'static',

		// The Static Countdown Date
		'value' => '',

		/**
		 * The timezone that will be used.
		 * 
		 * - server - Use server's timezone
		 * - client - Use client's timezone
		 */
		'timezone' => 'server',

		// Dynamic Days
		'dynamic_days' => 0,

		// Dynamic Hours
		'dynamic_hours' => 0,

		// Dynamic Minutes
		'dynamic_minutes' => 0,

		// Dynamic Seconds
		'dynamic_seconds' => 0,
		
		/**
		 * The countdown format.
		 * 
		 * Available tags:
		 * {years}
		 * {months}
		 * {days}
		 * {hours}
		 * {minutes}
		 * {seconds}
		 */
		'format' => '{days} days, {hours} hours, {minutes} minutes and {seconds} seconds',

		/**
		 * The countdown theme.
		 * 
		 * Available themes:
		 * theme1
		 * theme2
		 * theme3
		 */
		'theme' => 'theme1',

		/**
		 * Set the action once countdown finishes.
		 * 
		 * Available values:
		 * keep 	- Keep the countdown visible
		 * hide 	- Hide the countdown
		 * message	- Show a message
		 * redirect	- Redirect to a URL
		 */
		'countdown_action' => 'keep',

		/**
		 * The message appearing after the countdown has finished.
		 * 
		 * Requires `countdown_action` to be set to `message`
		 * 
		 * Example: Countdown finished.
		 */
		'finish_text' => '',

		/**
		 * The redirect URL once the countdown expires.
		 * 
		 * Requires `countdown_action` to be set to `redirect`
		 */
		'redirect_url' => '',

		// Whether to display Days
		'days' => true,

		// Days Label
		'days_label' => 'Days',
		
		// Whether to display Hours
		'hours' => true,

		// Hours Label
		'hours_label' => 'Hours',
		
		// Whether to display Minutes
		'minutes' => true,

		// Minutes Label
		'minutes_label' => 'Minutes',
		
		// Whether to display Seconds
		'seconds' => true,

		// Seconds Label
		'seconds_label' => 'Seconds',

		// Unit Border Color
		'item_border_color' => '',

		// Unit Label Color
		'unit_label_color' => '',

		// Digit Item Background Color. This applies for each of the 2 digits on a unit.
		'digit_background_color' => '',

		// Digit Item Text Color
		'digit_text_color' => '',

		// Unit Label Margin Top. The spacing between the unit and its label.
		'unit_label_margin_top' => '',

		// Digit Minimum Width. This is the minimum width for each number within a digit.
		'digit_min_width' => '',

		// Whether the units will appear in bold.
		'digits_bold' => false,

		// Whether each unit label will appear in bold.
		'unit_label_bold' => false,

		// Alignment
		'align' => '',

		// Extra attributes added to the widget
		'atts' => '',

		// CSS variables to style the countdown
		'css_vars' => ''
	];

	/**
	 * Class constructor
	 *
	 * @param array $options
	 */
	public function __construct($options = [])
	{
		parent::__construct($options);

		$this->prepare();
	}

	/**
	 * Prepares the countdown.
	 * 
	 * @return  void
	 */
	private function prepare()
	{
		$this->setCSSVars();

		$this->options['css_class'] .= ' ' . $this->options['theme'] . ' ' . $this->options['align'];

		if (!empty($this->options['value']) && $this->options['value'] !== '0000-00-00 00:00:00')
		{
			if ($this->options['countdown_type'] === 'static' && $this->options['timezone'] === 'server')
			{
				// Get timezone
				$tz = new \DateTimeZone(\JFactory::getApplication()->getCfg('offset', 'UTC'));

				// Convert given date time to UTC
				$this->options['value'] = date_create($this->options['value'], $tz)->setTimezone(new \DateTimeZone('UTC'))->format('c');
				
				// Apply server timezone
				$this->options['value'] = (new \DateTime($this->options['value']))->setTimezone($tz)->format('c');
			}
		}

		// Set countdown payload
		$payload = [
			'data-countdown-type="' . $this->options['countdown_type'] . '"',
			'data-value="' . $this->options['value'] . '"',
			'data-timezone="' . $this->options['timezone'] . '"',
			'data-dynamic-days="' . $this->options['dynamic_days'] . '"',
			'data-dynamic-hours="' . $this->options['dynamic_hours'] . '"',
			'data-dynamic-minutes="' . $this->options['dynamic_minutes'] . '"',
			'data-dynamic-seconds="' . $this->options['dynamic_seconds'] . '"',
			'data-finish-text="' . htmlspecialchars($this->options['finish_text']) . '"',
			'data-redirect-url="' . $this->options['redirect_url'] . '"',
			'data-theme="' . $this->options['theme'] . '"',
			'data-format="' . htmlspecialchars($this->options['format']) . '"',
			'data-countdown-action="' . $this->options['countdown_action'] . '"',
			'data-days="' . var_export($this->options['days'], true) . '"',
			'data-days-label="' . $this->options['days_label'] . '"',
			'data-hours="' . var_export($this->options['hours'], true) . '"',
			'data-hours-label="' . $this->options['hours_label'] . '"',
			'data-minutes="' . var_export($this->options['minutes'], true) . '"',
			'data-minutes-label="' . $this->options['minutes_label'] . '"',
			'data-seconds="' . var_export($this->options['seconds'], true) . '"',
			'data-seconds-label="' . $this->options['seconds_label'] . '"'
		];
		$this->options['atts'] = implode(' ', $payload);
	}

	/**
	 * Set widget CSS vars
	 * 
	 * @return  mixed
	 */
	private function setCSSVars()
	{
		$atts = [];

		if (!empty($this->options['item_border_color']))
		{
			$atts['item-border-color'] = $this->options['item_border_color'];
		}

		if (!empty($this->options['unit_label_color']))
		{
			$atts['unit-label-color'] = $this->options['unit_label_color'];
		}

		if (!empty($this->options['digit_background_color']))
		{
			$atts['digit-background-color'] = $this->options['digit_background_color'];
		}

		if (!empty($this->options['digit_text_color']))
		{
			$atts['digit-text-color'] = $this->options['digit_text_color'];
		}

		if (!empty($this->options['unit_label_margin_top']))
		{
			$atts['unit-label-margin-top'] = $this->options['unit_label_margin_top'] . 'px';
		}

		if (!empty($this->options['digit_min_width']))
		{
			$atts['digit-min-width'] = $this->options['digit_min_width'] . 'px';
		}

		if (!empty($this->options['digits_bold']))
		{
			$atts['digits-font-weight'] = 'bold';
		}

		if (!empty($this->options['unit_label_bold']))
		{
			$atts['unit-label-font-weight'] = 'bold';
		}

		if (empty($atts))
		{
			return;
		}

		if (!$css = \NRFramework\Helpers\CSS::cssVarsToString($atts, '.nrf-countdown.' . $this->options['id']))
		{
			return;
		}

		$this->options['css_vars'] = $css;
	}
}