<?php

/**
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2021 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

namespace NRFramework\Helpers\Widgets;

defined('_JEXEC') or die;

jimport('joomla.filesystem.file');
jimport('joomla.filesystem.folder');
jimport('joomla.filesystem.path');

use NRFramework\File;
use NRFramework\Mimes;
use NRFramework\Image;

class GalleryManager
{
	/**
	 * Upload file
	 *
	 * @param	array	$file						The request file as posted by form
	 * @param	string	$upload_settings			The upload settings
	 * @param	array	$media_uploader_file_data	Media uploader related file settings
	 * @param   array   $resizeSettings				The resize settings
	 *
	 * @return	mixed	String on success, Null on failure
	 */
	public static function upload($file, $upload_settings, $media_uploader_file_data, $resizeSettings)
	{
		$ds = DIRECTORY_SEPARATOR;
		
		// The source file name
		$source = '';

		// Move the image to the tmp folder
		try {
			$source = File::upload($file, null, $upload_settings['allowed_types'], $upload_settings['allow_unsafe']);
		} catch (\Throwable $th)
		{
			return false;
		}

		// If the file came from the Media Manager file and we are copying it, fix its filename
		if ($media_uploader_file_data['is_media_uploader_file'] && isset($media_uploader_file_data['copied_from_media_uploader']) && !$media_uploader_file_data['copied_from_media_uploader'])
		{
			$media_uploader_file_data['media_uploader_filename'] = self::getFilePathFromMediaUploaderFile($media_uploader_file_data['media_uploader_filename']);
		}

		// The uploaded file name
		$filename = null;

		// Whether to copy the base image
		$copyBaseImage = true;

		// If we are copying the base image, copy it to the temp folder.
		if ($copyBaseImage && !$source = File::move($source, $source, true))
		{
			return false;
		}

		// Check whether to copy and resize the original image
		if ($copyBaseImage && $resizeSettings['original_image_resize'])
		{
			$source = Image::resizeAndKeepAspectRatio($source, $resizeSettings['original_image_resize_width'], $resizeSettings['original_image_resize_quality']);
		}

		// Generate thumbnails
		if (!$thumb_data = self::generateThumbnail($source, $resizeSettings))
		{
			return false;
		}

		return [
			'filename' => is_null($filename) ? $thumb_data['filename'] : $filename,
			'thumbnail' => $thumb_data['resized_filename']
		];
	}
	
	/**
	 * Moves all given `tmp` items over to the destination folder.
	 * 
	 * @param   array   $items
	 * @param   string  $destination_folder
	 * 
	 * @return  mixed
	 */
	public static function moveTempItemsToDestination(&$items, $destination_folder)
	{
		if (!$destination_folder)
		{
			return;
		}

		// Create destination folder if missing
		if (!File::createDirs($destination_folder))
		{
			return;
		}

		$temp_folder = File::getTempFolder();

		// Move all files from `tmp` folder over to the `upload folder`
		foreach ($items as $key => &$item)
		{
			/**
			 * Skip invalid files.
			 * 
			 * These "fiels" can appear when we try to move files
			 * over to the destination folder when the gallery manager
			 * is still working to upload queueed files.
			 */
			if ($key === 'ITEM_ID')
			{
				continue;
			}
			
			$moved = false;
			
			// Ensure thumbnail in temp folder file exists
			if (file_exists($temp_folder . $item['thumbnail']))
			{
				// Move thumbnails
				$thumb = File::move($temp_folder . $item['thumbnail'], $destination_folder . $item['thumbnail']);
				$thumb_filename = pathinfo($thumb, PATHINFO_BASENAME);

				// If the moved file has changed name, use the new file name
				if ($item['thumbnail'] !== $thumb_filename)
				{
					$item['thumbnail'] = $thumb_filename;
				}

				$moved = true;
			}

			// Check if we have uploaded the full image as well and set it
			if ((($item['media_upload_source'] === 'false' && $item['copied_from_media_uploader'] === 'false') ||
				($item['media_upload_source'] === 'true' && $item['copied_from_media_uploader'] === 'true')) &&
				file_exists($temp_folder . $item['image']))
			{
				$image = File::move($temp_folder . $item['image'], $destination_folder . $item['image']);
				$image_filename = pathinfo($image, PATHINFO_BASENAME);

				// If the moved file has changed name, use the new file name
				if ($item['image'] !== $image_filename)
				{
					$item['image'] = $image_filename;
				}

				$moved = true;
			}

			if ($moved)
			{
				// Update destination path
				self::updateDestinationPath($item, $destination_folder);
			}
		}
	}

	/**
	 * Updates the destination path for the image and its thumbnail to the final destination folder.
	 * 
	 * @param   array   $item
	 * @param   string  $destination_folder
	 * 
	 * @return  mixed
	 */
	private static function updateDestinationPath(&$item, $destination_folder)
	{
		$ds = DIRECTORY_SEPARATOR;

		// Ensure destination folder is a relative path
		$destination_folder = ltrim(rtrim(str_replace(JPATH_ROOT, '', $destination_folder), $ds), $ds);

		$item['thumbnail'] = implode($ds, [$destination_folder, $item['thumbnail']]);

		$is_media_uploader_file = $item['media_upload_source'] === 'true';
		$copied_from_media_uploader = $item['copied_from_media_uploader'] === 'true';

		// Update the path of the original image only if copied from the Media Manager or uploaded manually
		if (!$is_media_uploader_file || ($is_media_uploader_file && $copied_from_media_uploader))
		{
			$item['image'] = implode($ds, [$destination_folder, $item['image']]);
		}
	}
	
	/**
	 * Media Uploader files look like: https://example.com/images/sampledata/parks/banner_cradle.png
	 * We remove the first part (https://example.com/images/) and keep the other part (relative path to image).
	 * 
	 * @param   string  $filename
	 * 
	 * @return  string
	 */
	private static function getFilePathFromMediaUploaderFile($filename)
	{
		$filenameArray = explode('images/', $filename, 2);
		unset($filenameArray[0]);
		$new_filepath = join($filenameArray);
		return 'images/' . $new_filepath;
	}

	/**
	 * Generates thumbnail
	 * 
	 * @param   string   $source			 	 Source image path.
	 * @param   array    $resizeSettings	 	 Resize Settings.
	 * @param   string   $destination_folder	 Destination folder.
	 * @param   boolean  $unique_filename		 Whether the thumbnails will have a unique filename.
	 * 
	 * @return  array
	 */
	public static function generateThumbnail($source, $resizeSettings, $destination_folder = null, $unique_filename = true)
	{
		$parts = pathinfo($source);
		$destination_folder = !is_null($destination_folder) ? $destination_folder : $parts['dirname'] . DIRECTORY_SEPARATOR;
		$destination = $destination_folder . $parts['filename'] . '_thumb.' . $parts['extension'];

		/**
		 * If height is zero, then we suppose we want to keep aspect ratio.
		 * 
		 * Resize with width & height: If thumbnail height is not set
		 * Resize and keep aspect ratio: If thumbnail height is set
		 */
		$resized_image = !is_null($resizeSettings['thumb_height']) && $resizeSettings['thumb_height'] !== '0'
			?
			Image::resize($source, $resizeSettings['thumb_width'], $resizeSettings['thumb_height'], $resizeSettings['thumb_resize_quality'], $resizeSettings['thumb_resize_method'], $destination, $unique_filename)
			:
			Image::resizeAndKeepAspectRatio($source, $resizeSettings['thumb_width'], $resizeSettings['thumb_resize_quality'], $destination, $unique_filename);
		
		if (!$resized_image)
		{
			return;
		}

		return [
			'filename' => basename($source),
			'resized_filename' => basename($resized_image)
		];
	}

	/**
	 * Deletes an uploaded file (resized original image and thumbnail).
	 *
	 * @param  string  $filename		The filename
	 * @param  string  $filename		The thumbnail filename
	 * @param  string  $upload_folder   The uploaded folder
	 * @param  array   $settings		Other settings we need for the deletion
	 *
	 * @return bool
	 */
	public static function deleteFile($filename, $thumbnail, $upload_folder, $settings)
	{
		if (empty($filename))
		{
			return false;
		}

		$is_media_uploader_file = $settings['is_media_uploader_file'];
		$copied_from_media_uploader = $settings['copied_from_media_uploader'];

		$deleted_original_image = false;

		// No upload folder needed when `upload_folder_type` is set to `auto` as we already have a relative path.
		if ($settings['upload_folder_type'] === 'auto')
		{
			$upload_folder = '';
		}

		// Delete the main image if it was added by the Media Uploader and copied over to destination folder or uploaded manually
		if (!$is_media_uploader_file || ($is_media_uploader_file && $copied_from_media_uploader))
		{
			$deleted_original_image = self::findAndDeleteFile($filename, $upload_folder);
		}

		// Check if we have a thumbnail and delete it
		$deleted_thumbnail = self::findAndDeleteFile($thumbnail, $upload_folder);

		return [
			'deleted_original_image' => $deleted_original_image,
			'deleted_thumbnail' => $deleted_thumbnail
		];
	}

	/**
	 * Tries to delete the file either from within the upload folder
	 * or within the tmp folder.
	 * 
	 * @param   string  $filename
	 * @param   string  $upload_folder
	 * 
	 * @return  mixed
	 */
	private static function findAndDeleteFile($filename, $upload_folder)
	{
		$ds = DIRECTORY_SEPARATOR;
		
		$file = \JPath::clean(implode($ds, [JPATH_ROOT, $upload_folder, $filename]));

		$deleted_file = false;
		
		// Try to delete it from the upload folder
		if (\JFile::exists($file)) {
			$deleted_file = \JFile::delete($file);
		}
		// Otherwise, try to delete it from the tmp folder
		else
		{
			$file = implode($ds, [File::getTempFolder(), $filename]);
			if (\JFile::exists($file))
			{
				$deleted_file = \JFile::delete($file);
			}
		}

		return $deleted_file;
	}
}